package com.swvoyages.reservation;

import java.net.URL;
import java.util.Iterator;
import java.util.Map;
import java.util.Vector;

import javax.wsdl.Definition;
import javax.wsdl.Import;
import javax.wsdl.factory.WSDLFactory;
import javax.wsdl.xml.WSDLReader;

import org.apache.soap.Constants;
import org.apache.soap.Fault;
import org.apache.soap.encoding.SOAPMappingRegistry;
import org.apache.soap.encoding.soapenc.BeanSerializer;
import org.apache.soap.rpc.Call;
import org.apache.soap.rpc.Parameter;
import org.apache.soap.rpc.Response;
import org.apache.soap.util.xml.QName;

/*
* La classe <code>ReservationService</code> implmente 
* l'interface <code>IReservationService</code>. Son objectif 
* consiste  offrir un service d'accs au systme de rservation 
* de la socit SW-Voyages.
*
* @version  1.0
*/
public class ReservationService implements IReservationService {

    /*
    * Prfixe associ  l'espace de noms du schma du 
    * service-type de rservation des partenaires.
    */
    private static String SCHEMA_NAMESPACE_PREFIX = "xsd1";

	/*
	* Nom du service type de rservation arienne.
	*/
	private static String AIR_TMODEL_NAME = "sw-aviation-xml-org:reservation";

	/*
	* Nom du service type de rservation automobile.
	*/
	private static String CAR_TMODEL_NAME = "sw-automobilisme-xml-org:reservation";

	/*
	* Nom du service type de rservation htelire.
	*/
	private static String HOTEL_TMODEL_NAME = "sw-hotellerie-xml-org:reservation";

	/*
	* URL du registre UDDI de rfrence.
	*/
	private static String UDDI_INQUIRE_URL = "http://www.sw-registre-uddi.org:8004/glue/inquire/uddi";

	/*
	* Le moteur de recherche des partenaires de rservation.
	*/
	private static ReservationFinder finder;
	
	/*
	* Le gestionnaire de toutes les rservations locales
	* en cours.
	*/
	private static ReservationManager manager;

	/*
	* L'espace de noms du schma du service-type de 
	* rservation de la centrale de rservation arienne.
	*/
	private static String airSchemaNamespace;

	/*
	* L'espace de noms du schma du service-type de 
	* rservation de la centrale de rservation automobile.
	*/
	private static String carSchemaNamespace;

	/*
	* L'espace de noms du schma du service-type de 
	* rservation de la centrale de rservation htelire.
	*/
	private static String hotelSchemaNamespace;

	/*
	* Le point d'accs du service de rservation de la 
	* centrale de rservation arienne.
	*/
	private static String airServiceAccessPoint;

	/*
	* Le point d'accs du service de rservation de la 
	* centrale de rservation automobile.
	*/
	private static String carServiceAccessPoint;

	/*
	* Le point d'accs du service de rservation de la 
	* centrale de rservation htelire.
	*/
	private static String hotelServiceAccessPoint;

	/*
	* Le registre des liaisons SOAP.
	*/
	private static SOAPMappingRegistry smr;

	/*
	* Le constructeur du service de rservation. Rcupre, 
	* via le registre UDDI, les URLs d'accs aux services 
	* de rservation des partenaires, ainsi que les espaces 
	* de noms associs aux schmas des services abstraits 
	* implments par les partenaires.
	*/
	public ReservationService() throws ReservationException {
		String[] parameters = getFinder().lookup(UDDI_INQUIRE_URL, AIR_TMODEL_NAME);
		setAirSchemaNamespace(getSchemaNamespace(parameters[0]));
		setAirServiceAccessPoint(parameters[1]);
		parameters = getFinder().lookup(UDDI_INQUIRE_URL, CAR_TMODEL_NAME);
		setCarSchemaNamespace(getSchemaNamespace(parameters[0]));
		setCarServiceAccessPoint(parameters[1]);
		parameters = getFinder().lookup(UDDI_INQUIRE_URL, HOTEL_TMODEL_NAME);
		setHotelSchemaNamespace(getSchemaNamespace(parameters[0]));
		setHotelServiceAccessPoint(parameters[1]);
	}

	/*
	* Renvoie l'espace de noms du schma du service-type de 
	* rservation de la centrale de rservation arienne.
	*
 	* @return		String l'espace de noms
	*/
	private String getAirSchemaNamespace() {
		return airSchemaNamespace;
	}

	/*
	* Affecte l'espace de noms du schma du service-type de 
	* rservation de la centrale de rservation arienne.
	*
	* @param		airSchemaNamespace l'espace de noms du 
	*				schma du service-type de rservation 
	*/
	private void setAirSchemaNamespace(String airSchemaNamespace) {
		this.airSchemaNamespace = airSchemaNamespace;
	}

	/*
	* Renvoie l'espace de noms du schma du service-type de 
	* rservation de la centrale de rservation automobile.
	*
 	* @return		String l'espace de noms
	*/
	private String getCarSchemaNamespace() {
		return carSchemaNamespace;
	}

	/*
	* Affecte l'espace de noms du schma du service-type de 
	* rservation de la centrale de rservation automobile.
	*
	* @param		carSchemaNamespace l'espace de noms du 
	*				schma du service-type de rservation 
	*/
	private void setCarSchemaNamespace(String carSchemaNamespace) {
		this.carSchemaNamespace = carSchemaNamespace;
	}

	/*
	* Renvoie l'espace de noms du schma du service-type de 
	* rservation de la centrale de rservation htelire.
	*
 	* @return		String l'espace de noms
	*/
	private String getHotelSchemaNamespace() {
		return hotelSchemaNamespace;
	}

	/*
	* Affecte l'espace de noms du schma du service-type de 
	* rservation de la centrale de rservation htelire.
	*
	* @param		hotelSchemaNamespace l'espace de noms du 
	*				schma du service-type de rservation 
	*/
	private void setHotelSchemaNamespace(String hotelSchemaNamespace) {
		this.hotelSchemaNamespace = hotelSchemaNamespace;
	}

	/*
	* Renvoie le point d'accs du service de rservation
	* de la centrale de rservation arienne.
	*
 	* @return		String le point d'accs du service
	*/
	private String getAirServiceAccessPoint() {
		return airServiceAccessPoint;
	}

	/*
	* Affecte le point d'accs du service de rservation
	* de la centrale de rservation arienne.
	*
	* @param		airServiceAccessPoint le point d'accs 
	*				du service de rservation
	*/
	private void setAirServiceAccessPoint(String airServiceAccessPoint) {
		this.airServiceAccessPoint = airServiceAccessPoint;
	}

	/*
	* Renvoie le point d'accs du service de rservation
	* de la centrale de rservation automobile.
	*
 	* @return		String le point d'accs du service
	*/
	private String getCarServiceAccessPoint() {
		return carServiceAccessPoint;
	}

	/*
	* Affecte le point d'accs du service de rservation
	* de la centrale de rservation automobile.
	*
	* @param		carServiceAccessPoint le point d'accs 
	*				du service de rservation
	*/
	private void setCarServiceAccessPoint(String carServiceAccessPoint) {
		this.carServiceAccessPoint = carServiceAccessPoint;
	}

	/*
	* Renvoie le point d'accs du service de rservation
	* de la centrale de rservation htelire.
	*
 	* @return		String le point d'accs du service
	*/
	private String getHotelServiceAccessPoint() {
		return hotelServiceAccessPoint;
	}

	/*
	* Affecte le point d'accs du service de rservation
	* de la centrale de rservation htelire.
	*
	* @param		hotelServiceAccessPoint le point d'accs 
	*				du service de rservation
	*/
	private void setHotelServiceAccessPoint(String hotelServiceAccessPoint) {
		this.hotelServiceAccessPoint = hotelServiceAccessPoint;
	}

	/*
	* Renvoie le registre des liaisons SOAP.
	*
 	* @return		SOAPMappingRegistry le registre des 
 	*				liaisons SOAP
	*/
	private synchronized SOAPMappingRegistry getSOAPMappingRegistry() throws ReservationException {
		if (smr==null) {
			smr = new SOAPMappingRegistry();
			BeanSerializer bs = new BeanSerializer();

			smr.mapTypes(Constants.NS_URI_SOAP_ENC,
				new QName(getAirSchemaNamespace(), "AirAvailability"),
				AirAvailability.class, bs, bs);
			smr.mapTypes(Constants.NS_URI_SOAP_ENC,
				new QName(getCarSchemaNamespace(), "CarAvailability"),
				CarAvailability.class, bs, bs);
			smr.mapTypes(Constants.NS_URI_SOAP_ENC,
				new QName(getHotelSchemaNamespace(), "HotelAvailability"),
				HotelAvailability.class, bs, bs);
		}
		return smr;
	}

	/*
	* Renvoie le moteur de recherche des partenaires de
	* rservation.
	*
 	* @return		ReservationFinder le singleton du moteur
 	*				de recherche des partenaires de rservation.
	*/
	private synchronized ReservationFinder getFinder() {
		if (finder==null)
			finder=ReservationFinder.getUniqueInstance() ;
		return finder;
	}

	/*
	* Renvoie le gestionnaire des rservations en cours.
	*
 	* @return		ReservationManager le singleton du
 	*				gestionnaire des rservations en cours
	*/
	private synchronized ReservationManager getManager() {
		if (manager==null) {
			manager=ReservationManager.getUniqueInstance();
		}
		return manager;
	}

	/*
	* Recherche de disponibilits pour la rservation d'un
	* voyage, en fonction des informations saisies par le client.
	*
	* @param		passengers le nombre de passagers
	* @param		from la localit de dpart
	* @param		to la localit d'arrive
	* @param		roundTrip aller retour/aller simple
	* @param		departureDay jour de dpart
	* @param		departureMonth mois de dpart 
	* @param		departureYear anne de dpart
	* @param		arrivalDay jour d'arrive
	* @param		arrivalMonth mois d'arrive 
	* @param		arrivalYear anne d'arrive
	* @param		hotelRequested rservation htelire demande
	* @param		carRequested rservation automobile demande
	*
 	* @return		int le n de rservation affect  la recherche
 	*				de disponibilits du client
	*/
	public int search(int passengers, String from, String to, boolean roundTrip, 
		int departureDay, int departureMonth, int departureYear, int arrivalDay, int arrivalMonth, 
		int arrivalYear, boolean hotelRequested, boolean carRequested) throws ReservationException {

		// Cration d'une nouvelle rservation SW-Voyages.
		Reservation reservation = new Reservation();

		reservation.setPassengers(passengers);
		reservation.setFrom(from.trim());
		reservation.setTo(to.trim());
		reservation.setRoundTrip(roundTrip);
		reservation.setDepartureDay(departureDay);
		reservation.setDepartureMonth(departureMonth);
		reservation.setDepartureYear(departureYear);
		reservation.setArrivalDay(arrivalDay);
		reservation.setArrivalMonth(arrivalMonth);
		reservation.setArrivalYear(arrivalYear);

		// Connexion au service de rservation de SW-Air.
		Vector params = new Vector();
		params.addElement(new Parameter("arg0", Integer.class, new Integer(passengers), null));
		params.addElement(new Parameter("arg1", String.class, from.trim(), null));
		params.addElement(new Parameter("arg2", String.class, to.trim(), null));
		params.addElement(new Parameter("arg3", Boolean.class, new Boolean(roundTrip), null));
		params.addElement(new Parameter("arg4", Integer.class, new Integer(departureDay), null));
		params.addElement(new Parameter("arg5", Integer.class, new Integer(departureMonth), null));
		params.addElement(new Parameter("arg6", Integer.class, new Integer(departureYear), null));
		params.addElement(new Parameter("arg7", Integer.class, new Integer(arrivalDay), null));
		params.addElement(new Parameter("arg8", Integer.class, new Integer(arrivalMonth), null));
		params.addElement(new Parameter("arg9", Integer.class, new Integer(arrivalYear), null));
		params.addElement(new Parameter("arg10", Integer.class, new Integer(reservation.getId()), null));

		Parameter result = invoke("search", params, getAirServiceAccessPoint(), null);
		int remoteAirReservationId = ((Integer)result.getValue()).intValue();
		reservation.setRemoteAirReservationId(remoteAirReservationId);

		// Connexion au service de rservation de SW-Voitures.
		if (carRequested) {
			result = invoke("search", params, getCarServiceAccessPoint(), null);
			int remoteCarReservationId = ((Integer)result.getValue()).intValue();
			reservation.setRemoteCarReservationId(remoteCarReservationId);
		}

		// Connexion au service de rservation de SW-Htels.
		if (hotelRequested) {
			result = invoke("search", params, getHotelServiceAccessPoint(), null);
			int remoteHotelReservationId = ((Integer)result.getValue()).intValue();
			reservation.setRemoteHotelReservationId(remoteHotelReservationId);
		}

		// Enregistrement de la nouvelle rservation dans le
		// registre local de SW-Voyages.
		int id = reservation.getId();
		getManager().newReservation((Object)new Integer(id), (Object)reservation);
		return id;
	}

	/*
	* Confirmation de la rservation dont l'identifiant est
	* pass en paramtre.
	*
	* @param		reservationId le n de la rservation 
	*				confirmer
	* @param		departureChoice l'identifiant de la
	*				disponibilit arienne (vol aller) choisie
	* @param		arrivalChoice l'identifiant de la disponibilit
	*				arienne (vol retour) choisie
	* @param		hotelChoice l'identifiant de la disponibilit
	*				htelire choisie
	* @param		carChoice l'identifiant de la disponibilit
	*				automobile choisie
	*
	* @return		<code>true</code> si la confirmation s'est
	*				bien passe; <code>false</code> sinon
	*/
	public boolean book(int reservationId, int departureChoice, int arrivalChoice,
		int hotelChoice, int carChoice) throws ReservationException {

	    // Recherche de la rservation locale
		Reservation reservation = getReservation(reservationId);

		// Connexion au service de rservation de SW-Air.
		int remoteId = reservation.getRemoteAirReservationId();
		int remoteDepartureAvailabilityId = reservation.getRemoteAirAvailabilityId(departureChoice);
		int remoteArrivalAvailabilityId = 0;
		if (reservation.getRoundTrip()) {
			remoteArrivalAvailabilityId = reservation.getRemoteAirAvailabilityId(arrivalChoice);
		}
		Vector params = new Vector();
		params.addElement(new Parameter("arg0", Integer.class, new Integer(remoteId), null));
		params.addElement(new Parameter("arg1", Integer.class, new Integer(remoteDepartureAvailabilityId), null));
		params.addElement(new Parameter("arg2", Integer.class, new Integer(remoteArrivalAvailabilityId), null));
		Parameter result = invoke("book", params, getAirServiceAccessPoint(), null);
		boolean booked = ((Boolean)result.getValue()).booleanValue();

		// Connexion au service de rservation de SW-Voitures.
		remoteId = reservation.getRemoteCarReservationId();
		if (remoteId!=0) {
			int remoteAvailabilityId = reservation.getRemoteCarAvailabilityId(carChoice);
			params = new Vector();
			params.addElement(new Parameter("arg0", Integer.class, new Integer(remoteId), null));
			params.addElement(new Parameter("arg1", Integer.class, new Integer(remoteAvailabilityId), null));
			result = invoke("book", params, getCarServiceAccessPoint(), null);
			booked = ((Boolean)result.getValue()).booleanValue();
		}

		// Connexion au service de rservation de SW-Htels.
		remoteId = reservation.getRemoteHotelReservationId();
		if (remoteId!=0) {
			int remoteAvailabilityId = reservation.getRemoteHotelAvailabilityId(hotelChoice);
			params = new Vector();
			params.addElement(new Parameter("arg0", Integer.class, new Integer(remoteId), null));
			params.addElement(new Parameter("arg1", Integer.class, new Integer(remoteAvailabilityId), null));
			result = invoke("book", params, getHotelServiceAccessPoint(), null);
			booked = ((Boolean)result.getValue()).booleanValue();
		}
		
		// Confirmation de la rservation locale (ici, la
		// la rservation est supprime, mais dans un systme
		// plus proche de la ralit, elle serait transmise
		//  un logiciel de back-office de SW-Voyages).
		getManager().removeReservation((Object)new Integer(reservationId));
		return true;
	}

	/*
	* Annulation de la rservation dont l'identifiant est
	* pass en paramtre.
	*
	* @param		reservationId le n de la rservation 
	*				annuler
	*
	* @return		<code>true</code> si l'annulation s'est
	*				bien passe; <code>false</code> sinon
	*/
	public boolean cancel(int reservationId)
		throws ReservationException {

	    // Recherche de la rservation locale
		Reservation reservation = getReservation(reservationId);

		// Connexion au service de rservation de SW-Air.
		int remoteId = reservation.getRemoteAirReservationId();
		Vector params = new Vector();
		params.addElement(new Parameter("arg0", Integer.class, new Integer(remoteId), null));
		Parameter result = invoke("cancel", params, getAirServiceAccessPoint(), null);
		boolean canceled = ((Boolean)result.getValue()).booleanValue();

		// Connexion au service de rservation de SW-Voitures.
		remoteId = reservation.getRemoteCarReservationId();
		if (remoteId!=0) {
			params = new Vector();
			params.addElement(new Parameter("arg0", Integer.class, new Integer(remoteId), null));
			result = invoke("cancel", params, getCarServiceAccessPoint(), null);
			canceled = ((Boolean)result.getValue()).booleanValue();
		}
		
		// Connexion au service de rservation de SW-Htels.
		remoteId = reservation.getRemoteHotelReservationId();
		if (remoteId!=0) {
			params = new Vector();
			params.addElement(new Parameter("arg0", Integer.class, new Integer(remoteId), null));
			result = invoke("cancel", params, getHotelServiceAccessPoint(), null);
			canceled = ((Boolean)result.getValue()).booleanValue();
		}

		// Annulation de la rservation locale.
		getManager().removeReservation((Object)new Integer(reservationId));
		return true;
	}

	/*
	* Renvoie la rservation locale dont l'identifiant
	* est pass en paramtre.
	*
	* @param		reservationId l'identifiant de la
	*				rservation locale
	*
 	* @return		Reservation la rservation locale dont
 	*				l'identifiant est pass en paramtre.
	*/
	private Reservation getReservation(int reservationId) {

		Reservation reservation = (Reservation)getManager().getReservation((Object)new Integer(reservationId));
		if (reservation == null) { 
			throw new IllegalArgumentException ("la rservation (id="+reservationId+") SW-Voyages n'est pas enregistre.");
		}
		return reservation;
	}

	/*
	* Renvoie les disponibilits de la centrale de rservation 
	* arienne pour la rservation locale dont l'identifiant
	* est pass en paramtre.
	*
	* @param		reservationId l'identifiant de la
	*				rservation locale
	*
 	* @return		AirAvailability[] les disponibilits de la
 	*				centrale de rservation arienne pour la 
 	*				rservation locale dont l'identifiant est 
 	*				pass en paramtre
	*/
	public AirAvailability[] getAirAvailabilities(int reservationId)
		throws ReservationException {

	    // Recherche de la rservation locale
		Reservation reservation = getReservation(reservationId);

		AirAvailability[] availabilities = reservation.getAirAvailabilities();
		if (availabilities.length!=0) {
			return availabilities;
		}

		// Connexion au service de rservation de SW-Air.
		Vector params = new Vector();
		params.addElement(new Parameter("arg0", Integer.class, new Integer(reservation.getRemoteAirReservationId()), null));

		Parameter result = invoke("getAirAvailabilities", params, getAirServiceAccessPoint(), getSOAPMappingRegistry());
		availabilities = (AirAvailability[])result.getValue();

		reservation.setAirAvailabilities(availabilities);
		return availabilities;
	}

	/*
	* Renvoie les disponibilits de la centrale de rservation 
	* automobile pour la rservation locale dont l'identifiant
	* est pass en paramtre.
	*
	* @param		reservationId l'identifiant de la
	*				rservation locale
	*
 	* @return		CarAvailability[] les disponibilits de la
 	*				centrale de rservation automobile pour la 
 	*				rservation locale dont l'identifiant est 
 	*				pass en paramtre.
	*/
	public CarAvailability[] getCarAvailabilities(int reservationId)
		throws ReservationException {

	    // Recherche de la rservation locale
		Reservation reservation = getReservation(reservationId);

		CarAvailability[] availabilities = reservation.getCarAvailabilities();
		if (availabilities.length!=0) {
			return availabilities;
		}

		// Connexion au service de rservation de SW-Voitures.
		Vector params = new Vector();
		params.addElement(new Parameter("arg0", Integer.class, new Integer(reservation.getRemoteCarReservationId()), null));

		Parameter result = invoke("getCarAvailabilities", params, getCarServiceAccessPoint(), getSOAPMappingRegistry());
		availabilities = (CarAvailability[])result.getValue();

		reservation.setCarAvailabilities(availabilities);
		return availabilities;
	}

	/*
	* Renvoie les disponibilits de la centrale de rservation 
	* htelire pour la rservation locale dont l'identifiant
	* est pass en paramtre.
	*
	* @param		reservationId l'identifiant de la
	*				rservation locale
	*
 	* @return		HotelAvailability[] les disponibilits de la
 	*				centrale de rservation htelire pour la 
 	*				rservation locale dont l'identifiant est 
 	*				pass en paramtre
	*/
	public HotelAvailability[] getHotelAvailabilities(int reservationId)
		throws ReservationException {

	    // Recherche de la rservation locale
		Reservation reservation = getReservation(reservationId);

		HotelAvailability[] availabilities = reservation.getHotelAvailabilities();
		if (availabilities.length!=0) {
			return availabilities;
		}

		// Connexion au service de rservation de SW-Htels.
		Vector params = new Vector();
		params.addElement(new Parameter("arg0", Integer.class, new Integer(reservation.getRemoteHotelReservationId()), null));

		Parameter result = invoke("getHotelAvailabilities", params, getHotelServiceAccessPoint(), getSOAPMappingRegistry());
		availabilities = (HotelAvailability[])result.getValue();

		reservation.setHotelAvailabilities(availabilities);
		return availabilities;
	}

	/*
	* Renvoie le rsultat de l'invocation du service de rservation
	* d'un partenaire.
	*
	* @param		methodName nom de la mthode du service 
	*				invoquer
	* @param		params vecteur des paramtres  fournir lors
	*				de l'invocation du service
	* @param		serviceAccessPoint point d'accs au service
	*				du partenaire
	* @param		smr registre de liaison SOAP  utiliser lors
	*				de l'invocation du service
	*
 	* @return		Parameter le rsultat de l'invocation du
 	*				service du partenaire
	*/
	private Parameter invoke(String methodName, Vector params,
		String serviceAccessPoint, SOAPMappingRegistry smr) throws ReservationException {

		Response response = null;
		try {
			// Construction de l'appel SOAP.
			Call call = new Call();
			if (smr!=null) {
				call.setSOAPMappingRegistry(smr);
			}
			call.setTargetObjectURI("urn:ReservationService");
			call.setMethodName(methodName);
			call.setEncodingStyleURI(Constants.NS_URI_SOAP_ENC);
			call.setParams(params);

			// Connexion au service du partenaire.
			response = call.invoke(new URL(serviceAccessPoint), "");
    
			// Vrification du rsultat de la connexion.
			if (response.generatedFault()) {
				Fault fault = response.getFault();
				StringBuffer sb = new StringBuffer();
				sb.append("Echec de l'invocation du service du partenaire :");
				sb.append(" Fault Code   = ");  
				sb.append(fault.getFaultCode());  
				sb.append(" Fault String = ");
				sb.append(fault.getFaultString());
				System.out.println(sb);
				throw new ReservationException(fault.getFaultString());
			} 
		}
		catch (Exception e) {
			StringBuffer sb = new StringBuffer();
			sb.append("une ");
			sb.append(e.getClass());
			sb.append(" s'est produite durant la connexion au service du partenaire");
			if ((e.getMessage()!=null) && (!(e.getMessage().equals("")))) {
				sb.append(" : ");
				sb.append(e.getMessage());
			}
			System.out.println(sb);
			throw new ReservationException(sb.toString(), e);
		}
		return response.getReturnValue();
	}

	/*
	* Renvoie l'espace de noms du schma XML associ au service-type 
	* de rservation d'un partenaire dont l'adresse de la description 
	* WSDL du service est passe en paramtre.
	*
	* @param		wsdlAccessPoint adresse de la description WSDL 
	*				du service de rservation du partenaire
	*
 	* @return		String l'espace de noms du schma XML associ au 
 	*				service-type de rservation du partenaire
	*/
	private String getSchemaNamespace(String wsdlAccessPoint) throws ReservationException {

		try {
			Definition definition = getServiceDefinition(wsdlAccessPoint, true);
			Map imports = definition.getImports();
			if (imports.isEmpty()) {
				throw new ReservationException("aucun import trouv");	
			}
			Iterator ii = imports.values().iterator();
			while (ii.hasNext()) {
				Vector iiv = (Vector)ii.next();
				String namespace = ((Import)iiv.firstElement()).getNamespaceURI();
				if (namespace==null) {
					throw new ReservationException("espace de noms du schma non trouv");	
				}
				return namespace;
			}
		}
		catch (Exception e) {
			e.printStackTrace();
		}
		throw new ReservationException("anomalie de rcupration de l'espace de noms du schma");
	}

	/*
	* Renvoie la dfinition de la description WSDL du service de 
	* rservation d'un partenaire dont l'adresse de la description 
	* WSDL du service est passe en paramtre.
	*
	* @param		wsdlAccessPoint adresse de la description WSDL 
	*				du service de rservation du partenaire
	* @param		imports spcifie si les documents imports par
	*				la description WSDL doivent tre lus ou non
	*
 	* @return		Definition la dfinition de la description WSDL 
 	*				du service de rservation du partenaire
	*/
	private Definition getServiceDefinition(String wsdlAccessPoint, 
		boolean imports) throws ReservationException {

		try {
			WSDLFactory factory = WSDLFactory.newInstance();
			WSDLReader reader = factory.newWSDLReader();
			reader.setFeature("javax.wsdl.verbose", false);
			reader.setFeature("javax.wsdl.importDocuments", imports);
			return reader.readWSDL(wsdlAccessPoint);
		}
		catch (Exception e) {
			e.printStackTrace();
			throw new ReservationException("description WSDL de service non trouve");
		}
	}
}